"use client";

import Link from "next/link";
import { useEffect, useMemo, useState } from "react";
import Papa from "papaparse";

function getKey() {
  try { return localStorage.getItem("ADMIN_KEY") || ""; } catch { return ""; }
}

export default function EventManage({ params }) {
  const key = useMemo(() => getKey(), []);
  const eventId = params.eventId;

  const [invites, setInvites] = useState(null);
  const [rsvps, setRsvps] = useState(null);
  const [sending, setSending] = useState(false);
  const [msg, setMsg] = useState("");
  const [err, setErr] = useState("");

  useEffect(() => { loadAll(); }, []);

  async function loadAll() {
    setErr("");
    const inv = await fetch(`/api/events/${eventId}/invites`, { headers: { "x-admin-key": key } });
    if (inv.ok) {
      const j = await inv.json();
      setInvites(j.invites || []);
    } else {
      setErr("S'ka autorizim ose s'u ngarkuan ftesat.");
      setInvites([]);
    }

    const rs = await fetch(`/api/events/${eventId}/rsvps`, { headers: { "x-admin-key": key } });
    if (rs.ok) {
      const j = await rs.json();
      setRsvps(j.rsvps || []);
    } else {
      setRsvps([]);
    }
  }

  async function onCsv(e) {
    const file = e.target.files?.[0];
    if (!file) return;

    setMsg("");
    setErr("");

    const text = await file.text();
    const parsed = Papa.parse(text, { header: true, skipEmptyLines: true });
    const guests = (parsed.data || []).map((r) => ({
      name: (r.name || r.emri || "").toString(),
      email: (r.email || "").toString(),
      phone: (r.phone || r.tel || r.telefon || "").toString(),
    })).filter(g => g.name.trim().length);

    if (!guests.length) { setErr("CSV s'ka rreshta valide. Duhet kolona: name,email,phone"); return; }

    const res = await fetch(`/api/events/${eventId}/guests`, {
      method: "POST",
      headers: { "content-type": "application/json", "x-admin-key": key },
      body: JSON.stringify({ guests }),
    });

    if (!res.ok) { setErr("S'u importuan mysafiret."); return; }

    const j = await res.json();
    setMsg(`U importuan: ${j.createdCount} mysafir. U krijuan ftesat automatikisht.`);
    await loadAll();
  }

  async function sendNow() {
    setSending(true);
    setMsg("");
    setErr("");
    try {
      const channels = {
        email: Boolean(document.getElementById("chEmail")?.checked),
        sms: Boolean(document.getElementById("chSms")?.checked),
        whatsapp: Boolean(document.getElementById("chWa")?.checked),
      };

      const subject = String(document.getElementById("subject")?.value || "Ftesa per dasme");
      const message = String(document.getElementById("message")?.value || "Ju lutem hapni ftesen ne linkun me poshte:");

      const res = await fetch(`/api/events/${eventId}/invites/send`, {
        method: "POST",
        headers: { "content-type": "application/json", "x-admin-key": key },
        body: JSON.stringify({ channels, subject, message, limit: 200 }),
      });

      const j = await res.json();
      if (!res.ok) { setErr("S'u derguan. Kontrollo konfigurimin."); return; }

      setMsg(`U procesuan: ${j.processed}. Konfigurim: email=${j.configured.email} twilio=${j.configured.twilio}`);
    } finally {
      setSending(false);
    }
  }

  return (
    <main className="min-h-screen bg-neutral-950 text-white">
      <div className="mx-auto max-w-5xl px-6 py-10">
        <div className="flex flex-wrap items-center justify-between gap-4">
          <div>
            <div className="text-xs tracking-widest text-neutral-400">MENAXHO EVENTIN</div>
            <h1 className="mt-2 text-2xl font-semibold">Ftesat & RSVP</h1>
            {err ? <div className="mt-2 text-sm text-red-300">{err}</div> : null}
            {msg ? <div className="mt-2 text-sm text-emerald-300">{msg}</div> : null}
          </div>
          <div className="flex gap-2">
            <Link href="/admin" className="rounded-xl border border-white/10 px-4 py-2 text-sm text-neutral-200 hover:bg-white/5">
              Mbrapa
            </Link>
            <Link href="/admin/events/new" className="rounded-xl bg-white px-4 py-2 text-sm font-semibold text-black hover:bg-neutral-200">
              + Event
            </Link>
          </div>
        </div>

        <div className="mt-8 grid gap-6 md:grid-cols-2">
          <Box title="Importo mysafire (CSV)">
            <div className="text-sm text-neutral-300">
              Kolonat minimale: <span className="font-semibold">name</span>, pastaj opsionale: <span className="font-semibold">email</span>, <span className="font-semibold">phone</span>.
            </div>
            <input
              type="file"
              accept=".csv,text/csv"
              onChange={onCsv}
              className="mt-4 block w-full rounded-xl border border-white/10 bg-black/30 px-3 py-2 text-sm text-neutral-200"
            />
          </Box>

          <Box title="Dergo njoftime">
            <div className="flex flex-wrap gap-3 text-sm">
              <label className="flex items-center gap-2">
                <input id="chEmail" type="checkbox" defaultChecked /> Email
              </label>
              <label className="flex items-center gap-2">
                <input id="chSms" type="checkbox" /> SMS
              </label>
              <label className="flex items-center gap-2">
                <input id="chWa" type="checkbox" /> WhatsApp
              </label>
            </div>

            <div className="mt-4 grid gap-3">
              <input
                id="subject"
                defaultValue="Ftesa per dasme"
                className="w-full rounded-xl border border-white/10 bg-black/40 px-3 py-2 text-sm text-white outline-none focus:border-white/25"
              />
              <textarea
                id="message"
                defaultValue="Ju lutem hapni ftesen ne linkun me poshte:"
                rows={3}
                className="w-full resize-none rounded-xl border border-white/10 bg-black/40 px-3 py-2 text-sm text-white outline-none focus:border-white/25"
              />
              <button
                disabled={sending}
                onClick={sendNow}
                className="rounded-xl bg-white px-4 py-2 text-sm font-semibold text-black hover:bg-neutral-200 disabled:opacity-60"
              >
                {sending ? "Duke derguar…" : "Dergo"}
              </button>
            </div>
          </Box>
        </div>

        <div className="mt-10 grid gap-6 md:grid-cols-2">
          <Box title={`Ftesat (${invites?.length ?? 0})`}>
            {!invites ? (
              <div className="text-sm text-neutral-300">Duke ngarkuar…</div>
            ) : invites.length === 0 ? (
              <div className="text-sm text-neutral-300">S'ka ftesa ende. Importo CSV.</div>
            ) : (
              <div className="space-y-3">
                {invites.slice(0, 50).map((i) => (
                  <div key={i.token} className="rounded-xl border border-white/10 bg-black/20 p-4">
                    <div className="text-sm font-semibold">{i.guest?.name || "—"}</div>
                    <div className="mt-1 text-xs text-neutral-400 break-all">{i.link}</div>
                    <div className="mt-2 flex gap-2">
                      <a className="rounded-lg bg-white/10 px-3 py-1 text-xs hover:bg-white/15" href={i.link} target="_blank" rel="noreferrer">
                        Hap
                      </a>
                      <button
                        className="rounded-lg border border-white/10 px-3 py-1 text-xs hover:bg-white/5"
                        onClick={() => navigator.clipboard?.writeText(i.link)}
                      >
                        Copy
                      </button>
                    </div>
                  </div>
                ))}
                {invites.length > 50 ? (
                  <div className="text-xs text-neutral-500">Po shfaqen vetem 50 te parat (per performance).</div>
                ) : null}
              </div>
            )}
          </Box>

          <Box title={`RSVP (${rsvps?.length ?? 0})`}>
            {!rsvps ? (
              <div className="text-sm text-neutral-300">Duke ngarkuar…</div>
            ) : rsvps.length === 0 ? (
              <div className="text-sm text-neutral-300">Asnje RSVP ende.</div>
            ) : (
              <div className="space-y-3">
                {rsvps.slice(0, 80).map((r) => (
                  <div key={r.token} className="rounded-xl border border-white/10 bg-black/20 p-4">
                    <div className="flex items-center justify-between gap-3">
                      <div className="text-sm font-semibold">{r.guest?.name || "—"}</div>
                      <div className="text-xs text-neutral-300">
                        {label(r.status)} · {r.count} persona
                      </div>
                    </div>
                    {r.note ? <div className="mt-2 text-xs text-neutral-400">{r.note}</div> : null}
                    <div className="mt-2 text-[11px] text-neutral-500">{new Date(r.updatedAt).toLocaleString("sq-AL")}</div>
                  </div>
                ))}
                {rsvps.length > 80 ? (
                  <div className="text-xs text-neutral-500">Po shfaqen vetem 80 te parat.</div>
                ) : null}
              </div>
            )}
          </Box>
        </div>

        <div className="mt-10">
          <button
            onClick={loadAll}
            className="rounded-xl border border-white/10 px-4 py-2 text-sm text-neutral-200 hover:bg-white/5"
          >
            Rifresko
          </button>
        </div>
      </div>
    </main>
  );
}

function Box({ title, children }) {
  return (
    <div className="rounded-3xl border border-white/10 bg-white/5 p-6">
      <div className="text-base font-semibold">{title}</div>
      <div className="mt-4">{children}</div>
    </div>
  );
}

function label(s) {
  if (s === "yes") return "Po";
  if (s === "no") return "Jo";
  return "Ndoshta";
}
